/*
 * Decompiled with CFR 0.152.
 */
package de.intarsys.tools.file;

import de.intarsys.tools.exception.ExceptionTools;
import de.intarsys.tools.file.PathTools;
import de.intarsys.tools.file.TempTools;
import de.intarsys.tools.stream.StreamTools;
import de.intarsys.tools.string.StringTools;
import de.intarsys.tools.system.SystemTools;
import de.intarsys.tools.valueholder.ObjectHolder;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.Charset;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileTools {
    public static final String DIRECTORY_LOCK = "directory.lock";
    private static final Logger Log = LoggerFactory.getLogger(FileTools.class);
    private static final Map<File, Lock> LOCKS = new HashMap<File, Lock>();

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static void appendFile(File source, File destination) throws IOException {
        if (FileTools.equals(source, destination)) {
            return;
        }
        FileInputStream is = null;
        FileOutputStream os = null;
        try {
            is = new FileInputStream(source);
            os = new FileOutputStream(destination, true);
            StreamTools.copy(is, os);
        }
        catch (IOException e) {
            try {
                throw e;
                catch (Exception e2) {
                    throw new IOException("copying failed (" + e2.getMessage() + ")");
                }
            }
            catch (Throwable throwable) {
                StreamTools.close(is);
                StreamTools.close(os);
                throw throwable;
            }
        }
        StreamTools.close(is);
        StreamTools.close(os);
    }

    public static File checkDirectory(File dir, boolean create, boolean checkCanRead, boolean checkCanWrite) throws IOException {
        if (dir == null) {
            return dir;
        }
        if (!dir.exists() && create) {
            FileTools.mkdirs(dir);
        }
        if (!dir.exists()) {
            throw new IOException("Can't create directory " + dir.getPath());
        }
        if (!dir.isDirectory()) {
            throw new IOException("Can't create directory " + dir.getPath());
        }
        if (checkCanRead && !dir.canRead()) {
            throw new IOException("No read access for directory " + dir.getPath());
        }
        if (checkCanWrite && !dir.canWrite()) {
            throw new IOException("No write access for directory " + dir.getPath());
        }
        return dir;
    }

    public static File checkDirectory(String path, boolean create, boolean checkCanRead, boolean checkCanWrite) throws IOException {
        if (StringTools.isEmpty(path)) {
            return null;
        }
        return FileTools.checkDirectory(new File(path), create, checkCanRead, checkCanWrite);
    }

    public static void copyBinaryFile(File source, File destination) throws IOException {
        if (destination.isDirectory()) {
            destination = new File(destination, source.getName());
        }
        Log.trace("copy binary '{}' to '{}'", (Object)source.getAbsolutePath(), (Object)destination.getAbsolutePath());
        if (destination.getParentFile() != null && !destination.getParentFile().exists()) {
            Log.trace("make directories for '{}'", (Object)destination.getAbsolutePath());
            FileTools.mkdirs(destination.getParentFile());
        }
        try {
            Files.copy(source.toPath(), destination.toPath(), StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.COPY_ATTRIBUTES);
        }
        catch (IOException e) {
            Log.debug("copy binary failed ({})", (Object)ExceptionTools.getMessage(e));
            throw new IOException("copy binary failed for '" + source.getAbsolutePath() + "' to '" + destination.getAbsolutePath() + "'", e);
        }
    }

    public static void copyFile(File source, File destination) throws IOException {
        FileTools.copyBinaryFile(source, destination);
    }

    /*
     * Loose catch block
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static void copyFile(File source, String sourceEncoding, File destination, String destinationEncoding) throws IOException {
        if (sourceEncoding == null || destinationEncoding == null || sourceEncoding.equals(destinationEncoding)) {
            FileTools.copyBinaryFile(source, destination);
            return;
        }
        Log.trace("copy encoded '{}' to '{}'", (Object)source.getAbsolutePath(), (Object)destination.getAbsolutePath());
        if (destination.isDirectory()) {
            destination = new File(destination, source.getName());
        }
        if (destination.getParentFile() != null && !destination.getParentFile().exists()) {
            Log.trace("make directories for '" + destination.getAbsolutePath() + "'");
            FileTools.mkdirs(destination.getParentFile());
        }
        BufferedInputStream is = null;
        BufferedOutputStream os = null;
        try {
            is = new BufferedInputStream(new FileInputStream(source));
            os = new BufferedOutputStream(new FileOutputStream(destination));
            StreamTools.copy(is, sourceEncoding, os, destinationEncoding);
        }
        catch (IOException e) {
            try {
                throw e;
                catch (Exception e2) {
                    Log.debug("copy encoded failed ({})", (Object)ExceptionTools.getMessage(e2));
                    throw new IOException("copy encoded failed for '" + source.getAbsolutePath() + "' to '" + destination.getAbsolutePath() + "'", e2);
                }
            }
            catch (Throwable throwable) {
                StreamTools.close(is);
                StreamTools.close(os);
                throw throwable;
            }
        }
        StreamTools.close(is);
        StreamTools.close(os);
        destination.setLastModified(source.lastModified());
    }

    public static void copyRecursively(File source, File destination) throws IOException {
        if (source.isFile()) {
            FileTools.copyFile(source, destination);
            return;
        }
        if (!source.isDirectory()) {
            throw new IOException("file '" + source.getAbsolutePath() + "' does not exist.");
        }
        if (destination.isFile()) {
            throw new IOException("cannot copy directory into file");
        }
        FileTools.mkdirs(destination);
        String[] content = source.list();
        if (content != null) {
            for (int i = 0; i < content.length; ++i) {
                FileTools.copyRecursively(new File(source, content[i]), new File(destination, content[i]));
            }
        }
        destination.setLastModified(source.lastModified());
    }

    public static File copyRecursivelyInto(File source, File destinationParent, String newName) throws IOException {
        if (destinationParent.isFile()) {
            throw new IOException("can't copy into file");
        }
        String destinationName = newName == null ? source.getName() : newName;
        File destinationFile = new File(destinationParent, destinationName);
        if (FileTools.equals(source, destinationFile)) {
            return destinationFile;
        }
        FileTools.copyRecursively(source, destinationFile);
        return destinationFile;
    }

    public static void createEmptyFile(File file) throws IOException {
        FileOutputStream os = new FileOutputStream(file);
        StreamTools.close(os);
    }

    public static File createTempFile(File file) throws IOException {
        String extension;
        Object name = file.getName();
        int index = ((String)name).lastIndexOf(46);
        if (index >= 0) {
            extension = ((String)name).substring(index);
            name = ((String)name).substring(0, index);
        } else {
            extension = "";
        }
        if (((String)name).length() < 3) {
            name = "tmp" + (String)name;
        }
        return TempTools.createTempFile((String)name, extension);
    }

    public static File createTempFile(String filename) throws IOException {
        return FileTools.createTempFile(new File(filename));
    }

    public static void delete(File file) throws IOException {
        if (!file.exists()) {
            Log.trace("delete '{}' skipped", (Object)file.getAbsolutePath());
            return;
        }
        Files.delete(file.toPath());
        Log.trace("deleted '{}'", (Object)file);
    }

    public static void deleteAfter(File directory, long millis, boolean recursiveScan) throws IOException {
        if (millis <= 0L) {
            return;
        }
        String[] fileNames = directory.list();
        if (fileNames == null) {
            throw new IOException("cannot list " + directory);
        }
        long checkMillis = System.currentTimeMillis() - millis;
        for (int j = 0; j < fileNames.length; ++j) {
            File file = new File(directory, fileNames[j]);
            if (file.isDirectory() && recursiveScan) {
                FileTools.deleteAfter(file, millis, recursiveScan);
            }
            if (file.lastModified() >= checkMillis) continue;
            Files.delete(file.toPath());
        }
    }

    public static void deleteEmptyDirectories(File root, File file) throws IOException {
        File[] files;
        if (file == null || FileTools.equals(file, root)) {
            return;
        }
        File dir = null;
        dir = file.isDirectory() ? file : file.getParentFile();
        if (dir != null && (files = dir.listFiles()) != null && files.length == 0) {
            try {
                Files.delete(dir.toPath());
                FileTools.deleteEmptyDirectories(root, dir.getParentFile());
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
    }

    public static void deleteRecursivly(File file) throws IOException {
        FileTools.deleteRecursivly(file, true);
    }

    public static void deleteRecursivly(File file, boolean deleteRoot) throws IOException {
        if (file == null || !file.exists()) {
            return;
        }
        if (file.isFile()) {
            Files.delete(file.toPath());
            return;
        }
        ObjectHolder<IOException> ex = new ObjectHolder<IOException>();
        try (DirectoryStream<Path> ds = Files.newDirectoryStream(file.toPath());){
            ds.forEach(path -> {
                try {
                    FileTools.deleteRecursivly(path.toFile());
                }
                catch (IOException e) {
                    ex.set(e);
                }
            });
        }
        if (deleteRoot) {
            try {
                Files.delete(file.toPath());
            }
            catch (IOException e) {
                ex.set(e);
            }
        }
        if (ex.get() != null) {
            throw (IOException)ex.get();
        }
    }

    public static boolean equals(File source, File destination) throws IOException {
        File canonicalDest;
        File canonicalSource;
        try {
            canonicalSource = source.getCanonicalFile();
        }
        catch (IOException e) {
            throw new IOException(e.getMessage() + " ('" + source.getAbsolutePath() + "')", e);
        }
        try {
            canonicalDest = destination.getCanonicalFile();
        }
        catch (IOException e) {
            throw new IOException(e.getMessage() + " ('" + destination.getAbsolutePath() + "')", e);
        }
        return canonicalSource.equals(canonicalDest);
    }

    public static String getBaseName(File file) {
        if (file == null) {
            return PathTools.getBaseName(null, null, "");
        }
        return PathTools.getBaseName(file.getName(), null, "");
    }

    public static String getBaseName(String filename) {
        return PathTools.getBaseName(filename);
    }

    public static String getBaseName(String filename, String extensionPrefix, String defaultName) {
        return PathTools.getBaseName(filename, extensionPrefix, defaultName);
    }

    public static byte[] getBytes(File file) throws IOException {
        byte[] byArray;
        FileInputStream is = null;
        try {
            is = new FileInputStream(file);
            byArray = StreamTools.getBytes(is);
        }
        catch (Throwable throwable) {
            StreamTools.close(is);
            throw throwable;
        }
        StreamTools.close(is);
        return byArray;
    }

    public static String getEncoding() {
        return System.getProperty("file.encoding");
    }

    public static String getExtension(File file) {
        return PathTools.getExtension(file.getName(), null, "");
    }

    public static String getExtension(String filename) {
        return PathTools.getExtension(filename);
    }

    public static String getExtension(String filename, String extensionPrefix, String defaultName) {
        return PathTools.getExtension(filename, extensionPrefix, defaultName);
    }

    public static String getFileName(File file) {
        return PathTools.getName(file.getName(), "");
    }

    public static String getFileName(String filename) {
        return PathTools.getName(filename);
    }

    public static String getFileName(String filename, String defaultName) {
        return PathTools.getName(filename, defaultName);
    }

    protected static File getLockFile(File file) {
        File lockFile = null;
        lockFile = !file.exists() || file.isFile() ? new File(file.getAbsolutePath() + ".lock") : new File(file, DIRECTORY_LOCK);
        return lockFile;
    }

    public static File getParentFile(File file) {
        File parentFile = file.getParentFile();
        if (parentFile == null) {
            parentFile = file.getAbsoluteFile().getParentFile();
        }
        if (parentFile == null) {
            return null;
        }
        return parentFile;
    }

    private static List getPathList(File f) throws IOException {
        ArrayList<String> l = new ArrayList<String>();
        for (File r = f.getCanonicalFile(); r != null; r = r.getParentFile()) {
            if (r.getName().length() == 0) {
                int dblptIndex = r.getPath().indexOf(":");
                if (dblptIndex == -1) {
                    l.add("");
                    continue;
                }
                l.add(r.getPath().substring(0, dblptIndex));
                continue;
            }
            l.add(r.getName());
        }
        ArrayList reversed = new ArrayList();
        for (int i = l.size() - 1; i >= 0; --i) {
            reversed.add(l.get(i));
        }
        return reversed;
    }

    public static String getPathName(File file) {
        return PathTools.getParent(file.getPath(), "");
    }

    public static String getPathName(String filename) {
        return PathTools.getParent(filename);
    }

    public static String getPathName(String filename, String defaultName) {
        return PathTools.getParent(filename, defaultName);
    }

    public static String getPathRelativeTo(File file, File base) throws IOException {
        String relativePath = null;
        if (base != null) {
            List fileList = FileTools.getPathList(file);
            List baseList = FileTools.getPathList(base);
            relativePath = FileTools.matchPathLists(fileList, baseList);
        }
        if (relativePath == null) {
            return file.getAbsolutePath();
        }
        return relativePath;
    }

    public static String getPathRelativeToIfAncestor(File file, File base) {
        if (base == null) {
            return file.getPath();
        }
        if (FileTools.isAncestor(base, file)) {
            try {
                return FileTools.getPathRelativeTo(file, base);
            }
            catch (IOException e) {
                return file.getPath();
            }
        }
        return file.getPath();
    }

    public static String getString(File file) throws IOException {
        return FileTools.getString(file, System.getProperty("file.encoding"));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String getString(File file, Charset charset) throws IOException {
        try (InputStream is = null;){
            is = new BufferedInputStream(new FileInputStream(file));
            String string = StreamTools.getString(is, charset);
            return string;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String getString(File file, String encoding) throws IOException {
        try (InputStream is = null;){
            is = new BufferedInputStream(new FileInputStream(file));
            String string = StreamTools.getString(is, encoding);
            return string;
        }
    }

    public static boolean isAncestor(File parent, File descendant) {
        if (parent == null) {
            return false;
        }
        File current = descendant;
        while (!parent.equals(current)) {
            if (current == null) {
                return false;
            }
            current = current.getParentFile();
        }
        return true;
    }

    public static boolean isExtensionMatch(File file, String extensions) {
        if (StringTools.isEmpty(extensions)) {
            return false;
        }
        String[] tempExtensions = extensions.toLowerCase().split(";");
        String tempName = file.getName().toLowerCase();
        for (int i = 0; i < tempExtensions.length; ++i) {
            if (!tempName.endsWith(tempExtensions[i])) continue;
            return true;
        }
        return false;
    }

    public static boolean isLegalPath(String filename) {
        if (SystemTools.isWindows()) {
            String fileName = new File(filename).getName();
            if (filename.indexOf("<") != -1 || filename.indexOf(">") != -1 || filename.indexOf("?") != -1 || filename.indexOf("\"") != -1 || fileName.indexOf(":") != -1 || filename.indexOf("|") != -1 || filename.indexOf("*") != -1) {
                return false;
            }
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean isLocked(File file) {
        Map<File, Lock> map = LOCKS;
        synchronized (map) {
            if (LOCKS.get(file) != null) {
                return true;
            }
            return FileTools.isLockedBasic(file);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean isLockedBasic(File file) {
        File lockFile = FileTools.getLockFile(file);
        if (lockFile.exists()) {
            Lock tempLock = null;
            try {
                tempLock = FileTools.lock(file);
                if (tempLock != null) {
                    Log.info("found abandoned lock '" + tempLock.file.getAbsolutePath() + "', take over");
                    boolean bl = false;
                    return bl;
                }
                boolean bl = true;
                return bl;
            }
            catch (Exception e) {
                boolean bl = true;
                return bl;
            }
            finally {
                if (tempLock != null) {
                    tempLock.release();
                }
            }
        }
        return false;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean isReadOnly(File file) {
        boolean bl;
        boolean delete = !file.exists();
        RandomAccessFile r = null;
        try {
            r = new RandomAccessFile(file, "rw");
            bl = false;
        }
        catch (IOException e) {
            boolean bl2;
            try {
                bl2 = true;
            }
            catch (Throwable throwable) {
                StreamTools.close(r);
                if (delete && file.exists()) {
                    file.delete();
                }
                throw throwable;
            }
            StreamTools.close(r);
            if (delete && file.exists()) {
                file.delete();
            }
            return bl2;
        }
        StreamTools.close(r);
        if (delete && file.exists()) {
            file.delete();
        }
        return bl;
    }

    public static boolean isWindows() {
        return File.separatorChar == '\\';
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Lock lock(File file) {
        Map<File, Lock> map = LOCKS;
        synchronized (map) {
            if (LOCKS.get(file) != null) {
                Log.trace("lock acquisition failed on " + file);
                return null;
            }
            Lock lock = FileTools.lockBasic(file);
            LOCKS.put(file, lock);
            return lock;
        }
    }

    public static Lock lockBasic(File file) {
        FileOutputStream os = null;
        FileChannel channel = null;
        File lockFile = FileTools.getLockFile(file);
        try {
            os = new FileOutputStream(lockFile);
            channel = os.getChannel();
            FileLock fileLock = channel.tryLock();
            if (fileLock != null) {
                lockFile.deleteOnExit();
                Lock lock = new Lock();
                lock.file = file;
                lock.lockFile = lockFile;
                lock.lockStream = os;
                Log.trace("lock acquired on {}", (Object)file);
                return lock;
            }
            throw new IOException("lock failed");
        }
        catch (Exception ex) {
            StreamTools.close(os);
            StreamTools.close(channel);
            Log.trace("lock acquisition failed on {}", (Object)file);
            return null;
        }
    }

    private static String matchPathLists(List fileList, List baseList) {
        Iterator sourceIterator = baseList.iterator();
        Iterator targetIterator = fileList.iterator();
        boolean intersection = false;
        while (sourceIterator.hasNext() && targetIterator.hasNext() && sourceIterator.next().equals(targetIterator.next())) {
            sourceIterator.remove();
            targetIterator.remove();
            intersection = true;
        }
        if (!intersection) {
            return null;
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < baseList.size(); ++i) {
            sb.append("../");
        }
        Iterator i = fileList.iterator();
        while (i.hasNext()) {
            sb.append(i.next());
            if (!i.hasNext()) continue;
            sb.append("/");
        }
        return sb.toString();
    }

    public static void mkdirs(File file) throws IOException {
        if (file == null) {
            return;
        }
        if (file.exists()) {
            return;
        }
        if (!file.mkdirs()) {
            try {
                Thread.sleep(50L);
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
            }
            if (file.exists()) {
                return;
            }
            if (!file.mkdirs()) {
                throw new IOException("failed to create " + file.getAbsolutePath());
            }
        }
    }

    public static void renameFile(File source, File destination) throws IOException {
        FileTools.renameFile(source, null, destination, null);
    }

    public static void renameFile(File source, String sourceEncoding, File destination, String destinationEncoding) throws IOException {
        boolean renameByCopy;
        if (FileTools.equals(source, destination)) {
            return;
        }
        try {
            Files.delete(destination.toPath());
        }
        catch (IOException iOException) {
            // empty catch block
        }
        boolean bl = renameByCopy = sourceEncoding != null && destinationEncoding != null && !sourceEncoding.equals(destinationEncoding);
        if (!renameByCopy) {
            Log.trace("rename '{}' to '{}'", (Object)source.getAbsolutePath(), (Object)destination.getAbsolutePath());
            if (source.renameTo(destination)) {
                return;
            }
            Log.trace("rename native failed");
        }
        File tempDestination = new File(destination.getAbsolutePath() + ".--temporary--");
        Log.debug("rename create temp file for '" + source.getAbsolutePath() + "' at '" + tempDestination.getAbsolutePath());
        FileTools.copyFile(source, sourceEncoding, tempDestination, destinationEncoding);
        if (!tempDestination.renameTo(destination)) {
            tempDestination.delete();
            throw new IOException("rename temp file failed for '" + tempDestination.getAbsolutePath() + "'");
        }
        if (!source.delete()) {
            destination.delete();
            throw new IOException("rename delete source failed for '" + source.getAbsolutePath() + "'");
        }
    }

    public static String replaceExtension(String filename, String extension) {
        if (StringTools.isEmpty(filename)) {
            return filename;
        }
        int dotPos = filename.lastIndexOf(46);
        if (dotPos >= 1) {
            filename = filename.substring(0, dotPos);
        }
        return filename + "." + extension;
    }

    public static File resolvePath(File parent, String path) {
        if (StringTools.isEmpty(path)) {
            return parent == null ? new File("") : parent;
        }
        if (parent == null) {
            return new File(path);
        }
        File file = new File(path);
        if (file.isAbsolute()) {
            return file;
        }
        return new File(parent, path);
    }

    @Deprecated
    public static byte[] toBytes(File file) throws IOException {
        return FileTools.getBytes(file);
    }

    @Deprecated
    public static String toString(File file) throws IOException {
        return FileTools.getString(file);
    }

    @Deprecated
    public static String toString(File file, String encoding) throws IOException {
        return FileTools.getString(file, encoding);
    }

    public static String trimPath(String param) {
        if (param == null) {
            return null;
        }
        String tmp = param.trim();
        String drivePrefix = "";
        if (tmp.length() >= 2 && tmp.charAt(1) == ':') {
            drivePrefix = tmp.substring(0, 2);
            tmp = tmp.substring(2);
        }
        tmp = tmp.replaceAll("[\\*\"\\?\\<\\>\\|\\:\\n\\t\\r\\f]", "_");
        return drivePrefix + tmp;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected static void unlock(Lock lock) {
        Map<File, Lock> map = LOCKS;
        synchronized (map) {
            LOCKS.remove(lock.file);
            lock.valid = false;
            StreamTools.close(lock.lockStream);
            try {
                Files.delete(lock.lockFile.toPath());
            }
            catch (IOException e) {
                Log.warn("failed to remove lock file {}", (Object)lock.lockFile, (Object)e);
            }
        }
    }

    public static void wait(File file, long timeout, long delay) throws IOException {
        long stop = System.currentTimeMillis() + timeout;
        try {
            while (true) {
                if (file.exists()) {
                    if (delay > 0L) {
                        long oldSize = -1L;
                        long newSize = file.length();
                        while (oldSize != newSize) {
                            oldSize = newSize;
                            Thread.sleep(delay);
                            newSize = file.length();
                        }
                    }
                    return;
                }
                if (System.currentTimeMillis() > stop) {
                    throw new IOException("timeout waiting for " + file.getPath());
                }
                Thread.sleep(1000L);
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("interrupted waiting for " + file.getPath());
        }
    }

    public static void write(File file, byte[] bytes) throws IOException {
        if (bytes == null) {
            throw new NullPointerException();
        }
        FileOutputStream os = new FileOutputStream(file);
        try {
            os.write(bytes);
        }
        finally {
            StreamTools.close(os);
        }
    }

    public static void write(File file, String text) throws IOException {
        FileTools.write(file, text, Charset.defaultCharset(), false);
    }

    public static void write(File file, String text, boolean append) throws IOException {
        FileTools.write(file, text, Charset.defaultCharset(), append);
    }

    public static void write(File file, String text, Charset charset) throws IOException {
        FileTools.write(file, text, charset, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void write(File file, String text, Charset charset, boolean append) throws IOException {
        FileOutputStream os = null;
        OutputStreamWriter writer = null;
        try {
            os = new FileOutputStream(file, append);
            writer = new OutputStreamWriter((OutputStream)os, charset);
            writer.write(text);
        }
        catch (Throwable throwable) {
            StreamTools.close(writer);
            StreamTools.close(os);
            throw throwable;
        }
        StreamTools.close(writer);
        StreamTools.close(os);
    }

    public static void write(File file, String text, String encoding) throws IOException {
        FileTools.write(file, text, encoding, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void write(File file, String text, String encoding, boolean append) throws IOException {
        if (text == null) {
            throw new NullPointerException();
        }
        FileOutputStream os = null;
        OutputStreamWriter writer = null;
        try {
            os = new FileOutputStream(file, append);
            writer = new OutputStreamWriter((OutputStream)os, encoding);
            writer.write(text);
        }
        catch (Throwable throwable) {
            StreamTools.close(writer);
            StreamTools.close(os);
            throw throwable;
        }
        StreamTools.close(writer);
        StreamTools.close(os);
    }

    private FileTools() {
    }

    public static class Lock {
        protected File file;
        protected File lockFile;
        protected FileOutputStream lockStream;
        protected boolean valid = true;

        protected Lock() {
        }

        public synchronized boolean isValid() {
            return this.valid;
        }

        public synchronized void release() {
            FileTools.unlock(this);
        }
    }
}

